module.exports = Trip;

var $ = require('jquery');
var TripParser = require('./trip.parser');
var TripUtils = require('./trip.utils');
var TripAnimation = require('./trip.animation');
var TripTheme = require('./trip.theme');
var TripConstant = require('./trip.constant');

/**
 * Trip
 *
 * @class Trip
 * @param {Array.<Object>} tripData
 * @param {Object} userOptions
 */
function Trip() {
  var noop = function() {};
  var userOptions;
  var tripData;
  var tripParser = new TripParser();

  // () - default parser mode without configurations
  if (arguments.length === 0) {
    tripData = tripParser.parse('default');
    userOptions = {};
  }
  else if (arguments.length === 1) {
    // ([]) - programming mode without configurations
    if (TripUtils.isArray(arguments[0])) {
      tripData = arguments[0];
      userOptions = {};
    }
    // ({}) - default parser mode with configurations
    else if (TripUtils.isObject(arguments[0])) {
      tripData = tripParser.parse('default');
      userOptions = arguments[0];
    }
    // ('.trip-nodes') - customized parser mode without configurations
    else if (TripUtils.isString(arguments[0])) {
      tripData = tripParser.parse(arguments[0]);
      userOptions = {};
    }
    // we don't support other formats here, so let's throw error here
    else {
      throw 'Please check documents for passing parameters, you may pass' +
        ' wrong parameters into constructor function !';
    }
  }
  // Users pass tripData directly from codebase
  else {
    // ([], {}) - programming mode with configurations
    if (TripUtils.isArray(arguments[0])) {
      tripData = arguments[0];
    }
    // ('.trip-nodes', {}) - customized parser mode with configurations
    else if (TripUtils.isString(arguments[0])) {
      tripData = tripParser.parse(arguments[0]);
    }
    userOptions = arguments[1];
  }

  /**
   * It is used to keep user and default settings.
   *
   * @memberOf Trip
   * @type {Object}
   */
  this.settings = $.extend({
    // basic config
    tripClass: '',
    tripIndex: 0,
    tripTheme: 'black',
    backToTopWhenEnded: false,
    overlayHolder: 'body',
    overlayZindex: 99999,
    delay: 1000,
    enableKeyBinding: true,
    enableAnimation: true,
    showSteps: false,
    showCloseBox: false,
    showHeader: false,
    skipUndefinedTrip: false,
    stopClickPropagation: false,

    // navigation
    showNavigation: false,
    canGoNext: true,
    canGoPrev: true,

    // labels
    nextLabel: 'Next',
    prevLabel: 'Back',
    finishLabel: 'Dismiss',
    closeBoxLabel: '&#215;',
    skipLabel: 'Skip',
    header: 'Step {{tripIndex}}',

    // callbacks for whole process
    onStart: noop,
    onEnd: noop,

    // callbacks for each trip
    onTripStart: noop,
    onTripEnd: noop,
    onTripStop: noop,
    onTripPause: noop,
    onTripResume: noop,
    onTripChange: noop,
    onTripClose: noop,

    // animation
    animation: 'fadeIn'
  }, userOptions);

  if (!this.settings.tripBlockHTML) {
    var html = TripTheme.get(this.settings.tripTheme);
    if (!html) {
      html = TripTheme.get('default');
    }
    this.settings.tripBlockHTML = html;
  }

  this.tripData = tripData;

  // used SELs
  this.$tripBlock = null;
  this.$overlay = null;
  this.$bar = null;
  this.$root = $('html');

  // save the current trip index
  this.tripDirection = 'next';
  this.timer = null;
  this.progressing = false;
  this.hasExposedElements = false;

  // for testing
  this.CONSTANT = TripConstant;
}

Trip.prototype = {
  /**
   * Expose element which has hasExposedElements property.
   *
   * @memberOf Trip
   * @type {Funtion}
   */
  showExposedElements: function() {
    var o = this.getCurrentTripObject();
    var oldCSS;
    var newCSS;
    var $sel;

    if (typeof o.expose === 'string') {
      $sel = $(o.expose);
    }
    else if (o.expose instanceof $) {
      $sel = o.expose;
    }
    else {
      $sel = $(o.sel);
    }

    this.hasExposedElements = true;

    // NOTE: issue #68
    // we have to make sure $sel does exist because we may have no
    // $sel when using special directions
    if ($sel.get(0) !== undefined) {
      oldCSS = {
        position: $sel.css('position'),
        zIndex: $sel.css('z-Index')
      };

      newCSS = {
        position: (function() {
          // NOTE: issue #63
          // We can't direclty use 'relative' if the original element
          // is using properties other than 'relative' because
          // this would break the UI.
          if (['absolute', 'fixed'].indexOf(oldCSS.position) > -1) {
            return oldCSS.position;
          }
          else {
            return 'relative';
          }
        }()),
        // we have to make it higher than the overlay
        zIndex: this.settings.overlayZindex + 1
      };

      $sel
        .data('trip-old-css', oldCSS)
        .css(newCSS)
        .addClass('trip-exposed');
    }
  },

  /**
   * Make exposed element back to normal state and hide overlay.
   *
   * @memberOf Trip
   * @type {Funtion}
   */
  hideExposedElements: function() {
    var $exposedSel = $('.trip-exposed');
    this.hasExposedElements = false;

    // NOTE: issue #68
    // we have to make sure $sel does exist because we may have no
    // $sel when using special directions
    if ($exposedSel.get(0) !== undefined) {
      var oldCSS = $exposedSel.data('trip-old-css');

      $exposedSel
        .css(oldCSS)
        .removeClass('trip-exposed');
    }
  },

  /**
   * When users resize its browser, we will rerun Trip and restart the timer.
   * TODO: We have to debounce this function later to make performance better.
   *
   * @memberOf Trip
   * @type {Function}
   */
  bindResizeEvents: function() {
    var that = this;
    var timer;

    $(window).on('resize.Trip', function() {
      window.clearTimeout(timer);
      timer = window.setTimeout(function() {
        that.run();
      }, TripConstant.RESIZE_TIMEOUT);
    });
  },

  /**
   * Remove resize event from window.
   *
   * @memberOf Trip
   * @type {Function}
   */
  unbindResizeEvents: function() {
    $(window).off('resize.Trip');
  },

  /**
   * Bind keydown events on document.
   *
   * @memberOf Trip
   * @type {Function}
   */
  bindKeyEvents: function() {
    var that = this;
    $(document).on({
      'keydown.Trip': function(e) {
        // `this` will be bound to #document DOM element here
        that.keyEvent.call(that, e);
      }
    });
  },

  /**
   * Remove keydown events from document.
   *
   * @memberOf Trip
   * @type {Function}
   */
  unbindKeyEvents: function() {
    $(document).off('keydown.Trip');
  },

  /**
   * Bound keydown events. We will do specific actions when matched keys
   * are pressed by user.
   *
   * @memberOf Trip
   * @type {function}
   * @param {Event} e
   */
  keyEvent: function(e) {
    switch (e.which) {
      case TripConstant.ESC:
        this.stop();
        break;

      case TripConstant.SPACE:
        // space will make the page jump
        e.preventDefault();
        this.pause();
        break;

      case TripConstant.LEFT_ARROW:
      case TripConstant.UP_ARROW:
        this.prev();
        break;

      case TripConstant.RIGHT_ARROW:
      case TripConstant.DOWN_ARROW:
        this.next();
        break;
    }
  },

  /**
   * Stop API, which will stop the trip.
   *
   * @memberOf Trip
   * @type {Function}
   * @public
   */
  stop: function() {
    if (this.timer) {
      this.timer.stop();
      this.timer = null;
    }

    if (this.hasExposedElements) {
      this.hideExposedElements();
      this.toggleExposedOverlay(false);
    }

    this.hideTripBlock();
    this.unbindKeyEvents();
    this.unbindResizeEvents();

    var tripObject = this.getCurrentTripObject();
    if (tripObject.nextClickSelector) {
      $(tripObject.nextClickSelector).off('click.Trip');
    }

    var tripStop = tripObject.onTripStop || this.settings.onTripStop;
    tripStop.call(this, this.tripIndex, tripObject);

    this.settings.onEnd(this.tripIndex, tripObject);

    // reset tripIndex when stopped
    this.setIndex(this.settings.tripIndex);
  },

  /**
   * This is an wrapper for pause and resume API.
   *
   * @memberOf Trip
   * @type {Function}
   */
  pauseOrResume: function() {
    if (!this.timer) {
      return;
    }

    if (this.progressing) {
      this.timer.pause();
      this.pauseProgressBar();
    }
    else {
      var remainingTime = this.timer.resume();
      this.resumeProgressBar(remainingTime);
    }

    this.progressing = !this.progressing;
  },

  /**
   * pause API, which will pause the trip.
   *
   * @memberOf Trip
   * @type {Function}
   * @public
   */
  pause: function() {
    this.pauseOrResume();
    var tripObject = this.getCurrentTripObject();
    var tripPause = tripObject.onTripPause || this.settings.onTripPause;
    tripPause.call(this, this.tripIndex, tripObject);
  },

  /**
   * pause API, which will pause the trip.
   *
   * @memberOf Trip
   * @type {Function}
   * @public
   */
  resume: function() {
    this.pauseOrResume();
    var tripObject = this.getCurrentTripObject();
    var tripResume = tripObject.onTripResume || this.settings.onTripResume;
    tripResume.call(this, this.tripIndex, tripObject);
  },

  /**
   * next API, which will jump to next the trip.
   *
   * @memberOf Trip
   * @param {Number} tripIndex
   * @type {Function}
   * @public
   */
  next: function(tripIndex) {
    var that = this;
    var useDifferentIndex = !isNaN(tripIndex);

    // If we do give `tripIndex` here, it means that we want to directly jump
    // to that index no matter how. So in that case, ignore `canGoNext` check.
    if (!useDifferentIndex && !this.canGoNext()) {
      return;
    }

    this.tripDirection = 'next';

    // This is te best timing to call tripEnd because no matter
    // users use arrow key or trip was changed by timer, we will
    // all be here.
    var tripObject = this.getCurrentTripObject();
    var tripEnd = tripObject.onTripEnd || this.settings.onTripEnd;
    var tripEndDefer = tripEnd.call(this, this.tripIndex, tripObject);

    $.when(tripEndDefer).then(function() {
      if (useDifferentIndex) {
        if (that.timer) {
          that.timer.stop();
        }
        that.setIndex(tripIndex);
        that.run();
        return;
      }

      if (that.isLast()) {
        that.doLastOperation();
      }
      else {
        that.increaseIndex();
        that.run();
      }
    });
  },

  /**
   * prev API, which will jump to previous trip.
   *
   * @memberOf Trip
   * @type {Function}
   * @public
   */
  prev: function() {
    var that = this;

    if (!this.canGoPrev()) {
      return;
    }

    this.tripDirection = 'prev';

    // When this is executed, it means users click on the arrow key to
    // navigate back to previous trip. In that scenario, this is the better
    // place to call onTripEnd before modifying tripIndex.
    var tripObject = this.getCurrentTripObject();
    var tripEnd = tripObject.onTripEnd || this.settings.onTripEnd;
    var tripEndDefer = tripEnd(this.tripIndex, tripObject);

    $.when(tripEndDefer).then(function() {
      if (!that.isFirst()) {
        that.decreaseIndex();
      }
      that.run();
    });
  },

  /**
   * Show current trip. In this method, we will control all stuffs about
   * current trip including animation, timer, expose, progress bar.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  showCurrentTrip: function(o) {
    this.removeAnimation();

    // preprocess when we have to show trip block
    if (this.timer) {
      this.timer.stop();
      this.timer = null;
    }

    if (this.hasExposedElements) {
      this.hideExposedElements();

      if (!o.expose) {
        this.toggleExposedOverlay(false);
      }
    }

    if (this.progressing) {
      this.hideProgressBar();

      // not doing the progress effect
      this.progressing = false;
    }

    this.setTripBlock(o);
    this.showTripBlock(o);

    if (TripUtils.isSet(o.enableAnimation, this.settings.enableAnimation)) {
      this.addAnimation(o);
    }

    if (o.expose) {
      this.showExposedElements();
      this.toggleExposedOverlay(true);
    }
  },

  /**
   * This is the last operation when we successfully finish all trips in
   * the end.
   *
   * @memberOf Trip
   * @type {Function}
   */
  doLastOperation: function() {
    if (this.timer) {
      this.timer.stop();
      this.timer = null;
    }

    if (this.settings.enableKeyBinding) {
      this.unbindKeyEvents();
    }

    this.hideTripBlock();
    this.unbindResizeEvents();

    if (this.hasExposedElements) {
      this.hideExposedElements();
      this.toggleExposedOverlay(false);
    }

    if (this.settings.backToTopWhenEnded) {
      this.$root.animate({ scrollTop: 0 }, 'slow');
    }

    var tripObject = this.getCurrentTripObject();
    this.settings.onEnd(this.tripIndex, tripObject);

    // reset tripIndex when finished
    this.setIndex(this.settings.tripIndex);
  },

  /**
   * This is used to show progress bar UI. We will use jQuery to manipulate
   * the animation.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Number} delay
   */
  showProgressBar: function(delay) {
    var that = this;

    this.$bar.animate({
      width: '100%'
    }, delay, 'linear', function() {
      that.$bar.width(0);
    });
  },

  /**
   * Hide the progress bar and stop animations.
   *
   * @memberOf Trip
   * @type {Function}
   */
  hideProgressBar: function() {
    this.$bar.width(0);
    this.$bar.stop(true);
  },

  /**
   * Pause the progress bar.
   *
   * @memberOf Trip
   * @type {Function}
   */
  pauseProgressBar: function() {
    this.$bar.stop(true);
  },

  /**
   * Resumse the progress bar.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Number} remainingTime
   */
  resumeProgressBar: function(remainingTime) {
    this.showProgressBar(remainingTime);
  },

  /**
   * This is the main function to control each trip. In this method, we will
   * make sure every tripData is valid and use that to do following works like
   * showing trip, setup timer and trigger registered callbacks at the right
   * timing.
   *
   * @memberOf Trip
   * @type {Function}
   */
  run: function() {
    var that = this;
    var tripObject = this.getCurrentTripObject();
    var tripStart = tripObject.onTripStart || this.settings.onTripStart;
    var tripChange = tripObject.onTripChange || this.settings.onTripChange;
    var delay = tripObject.delay || this.settings.delay;

    if (!this.isTripDataValid(tripObject)) {
      // force developers to double check tripData again
      if (this.settings.skipUndefinedTrip === false) {
        TripUtils.log('Your tripData is not valid at index: ' + this.tripIndex);
        this.stop();
        return false;
      }
      // let it go
      else {
        return this[this.tripDirection]();
      }
    }

    this.showCurrentTrip(tripObject);
    this.showProgressBar(delay);
    this.progressing = true;

    tripChange.call(this, this.tripIndex, tripObject);
    tripStart.call(this, this.tripIndex, tripObject);

    // set timer to show next, if the timer is less than zero we expect
    // it to be manually advanced
    if (delay >= 0) {
      this.timer = new TripUtils.Timer(function() {
        that.next();
      }, delay);
    }
  },

  /**
   * Check whether current trip is the first one.
   *
   * @memberOf Trip
   * @type {Function}
   * @return {Boolean} whether current trip is the first one
   */
  isFirst: function() {
    return (this.tripIndex === 0) ? true : false;
  },

  /**
   * Check whether current trip is the last one.
   *
   * @memberOf Trip
   * @type {Function}
   * @return {Boolean} whether current trip is the last one
   */
  isLast: function() {
    return (this.tripIndex === this.tripData.length - 1) ? true : false;
  },

  /**
   * Check whether tripData is valid
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o tripData
   * @return {Boolean} whether tripData is valid
   */
  isTripDataValid: function(o) {
    if (this.hasSpecialDirections()) {
      return true;
    }

    if (o.nextClickSelector && $(o.nextClickSelector).length === 0) {
      return false;
    }

    // have to check `sel` & `content` two required fields
    if (typeof o.content === 'undefined' ||
      typeof o.sel === 'undefined' ||
      o.sel === null ||
      o.sel.length === 0 ||
      $(o.sel).length === 0) {
        return false;
    }
    return true;
  },

  /**
   * Check whether position is special or not
   *
   * @memberOf Trip
   * @type {Function}
   * @param {String} position position
   * @return {Boolean} whether position is speical direction or not
   */
  hasSpecialDirections: function() {
    var o = this.getCurrentTripObject();
    var position = o.position;
    var specialDirections = [
      'screen-ne',
      'screen-se',
      'screen-sw',
      'screen-nw',
      'screen-center'
    ];

    // if we have set special direction,
    // we don't need to check sel
    if ($.inArray(position, specialDirections) >= 0) {
      return true;
    }
    return false;
  },

  /**
   * Check whether we can go to previous trip or not.
   *
   * @memberOf Trip
   * @type {Function}
   * @return {Boolean} whether we can go to previous trip
   */
  canGoPrev: function() {
    var tripObject = this.getCurrentTripObject();
    var canGoPrev = tripObject.canGoPrev;

    if (typeof canGoPrev === 'undefined') {
      canGoPrev = this.settings.canGoPrev;
    }

    if (typeof canGoPrev === 'function') {
      canGoPrev = canGoPrev.call(this, this.tripIndex, tripObject);
    }

    // For this special case, there is no need to let users go back to previous
    // state.
    if (this.tripIndex === 0) {
      canGoPrev = false;
    }

    return canGoPrev;
  },

  /**
   * Check whether we can go to next trip or not.
   *
   * @memberOf Trip
   * @type {Function}
   * @return {Boolean} whether we can go to next trip
   */
  canGoNext: function() {
    var tripObject = this.getCurrentTripObject();
    var canGoNext = tripObject.canGoNext;

    if (typeof canGoNext === 'undefined') {
      canGoNext = this.settings.canGoNext;
    }

    if (typeof canGoNext === 'function') {
      canGoNext = canGoNext.call(this, this.tripIndex, tripObject);
    }

    return canGoNext;
  },

  setIndex: function(tripIndex) {
    tripIndex = Math.max(0, Math.min(tripIndex, this.tripData.length - 1));
    this.tripIndex = tripIndex;

    // reflect the trip information on UI
    this.$tripBlock.attr('data-trip-step', this.tripIndex);
  },

  /**
   * We can call this method to increase tripIndex because we are not allowed
   * to manipualte the value directly.
   *
   * @memberOf Trip
   * @type {Function}
   */
  increaseIndex: function() {
    this.setIndex(this.tripIndex + 1);
  },

  /**
   * We can call this method to decrease tripIndex because we are not allowed
   * to manipualte the value directly.
   *
   * @memberOf Trip
   * @type {Function}
   */
  decreaseIndex: function() {
    this.setIndex(this.tripIndex - 1);
  },

  /**
   * This method is used to get current trip data.
   *
   * @memberOf Trip
   * @type {Function}
   * @return {Object} current trip data
   */
  getCurrentTripObject: function() {
    return this.tripData[this.tripIndex];
  },

  /**
   * This method is used to replace all passed content with `tripIndex` and
   * `tripTotal` information.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {String} content
   * @return {String} replaced content
   */
  getReplacedTripContent: function(content) {
    content = content || '';
    var reTripIndex = /\{\{(tripIndex)\}\}/g;
    var reTripTotal = /\{\{(tripTotal)\}\}/g;

    content = content.replace(reTripIndex, this.tripIndex + 1);
    content = content.replace(reTripTotal, this.tripData.length);
    return content;
  },

  /**
   * Based on current trip data, we will use this method to set all stuffs
   * we want like content, prev / next labels, close button, used animations.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  setTripBlock: function(o) {
    var $tripBlock = this.$tripBlock;
    var that = this;

    // toggle used settings
    var showCloseBox =
          TripUtils.isSet(o.showCloseBox, this.settings.showCloseBox);
    var showNavigation =
          TripUtils.isSet(o.showNavigation, this.settings.showNavigation);
    var showHeader = TripUtils.isSet(o.showHeader, this.settings.showHeader);
    var showSteps = TripUtils.isSet(o.showSteps, this.settings.showSteps);

    // labels
    var closeBoxLabel = o.closeBoxLabel || this.settings.closeBoxLabel;
    var prevLabel = o.prevLabel || this.settings.prevLabel;
    var nextLabel = o.nextLabel || this.settings.nextLabel;
    var finishLabel = o.finishLabel || this.settings.finishLabel;
    var skipLabel = o.skipLabel || this.settings.skipLabel;

    // other user customized contents
    var header = o.header || this.settings.header;

    $tripBlock
      .find('.trip-header')
        .html(this.getReplacedTripContent(header))
        .toggle(showHeader);

    $tripBlock
      .find('.trip-content')
        .html(this.getReplacedTripContent(o.content));

    $tripBlock
      .find('.trip-prev')
        .toggleClass('disabled', !this.canGoPrev())
        .html(prevLabel)
        .toggle(showNavigation);

    $tripBlock
      .find('.trip-next')
        .toggleClass('disabled', !this.canGoNext())
        .html(this.isLast() ? finishLabel : nextLabel)
        .toggle(showNavigation && !o.nextClickSelector);

    $tripBlock
      .find('.trip-skip')
        .html(skipLabel)
        .toggle(showNavigation);

    $tripBlock
      .find('.trip-close')
        .html(closeBoxLabel)
        .toggle(showCloseBox);

    $tripBlock
      .find('.trip-progress-steps')
        .toggle(showSteps)
      .find('.trip-progress-step')
        .not(':eq(' + this.tripIndex + ')')
          .removeClass('selected')
          .end()
        .eq(this.tripIndex)
          .addClass('selected');

    // remove old styles then add new one
    $tripBlock.removeClass(
      'e s w n screen-ne screen-se screen-sw screen-nw screen-center');
    $tripBlock.addClass(o.position);

    // if we have a nextClickSelector use that as the trigger for
    // the next button
    if (o.nextClickSelector) {
      $(o.nextClickSelector).off('click.Trip');
      $(o.nextClickSelector).one('click.Trip', function(e) {
        e.preventDefault();
        // Force IE/FF to lose focus
        $(this).blur();
        that.next();
      });
    }

    // NOTE: issue #27
    // we have to set position left first then position top
    // because when tripBlock hits the page margin, it will become
    // multi-lined and this will break cached attributes.
    //
    // In this way, we have to count these attributes at runtime.
    this.setTripBlockPosition(o, 'horizontal');
    this.setTripBlockPosition(o, 'vertical');
  },

  /**
   * This method is mainly used to help us position the trip block. As you can
   * see, we will find out the $sel and its positions first then put our trip
   * block at the right location.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   * @param {String} horizontalOrVertical
   */
  setTripBlockPosition: function(o, horizontalOrVertical) {
    var $tripBlock = this.$tripBlock;

    //Styles need to be reset before capturing outer height and width
    if (horizontalOrVertical === 'horizontal') {
        // reset styles first
        $tripBlock.css({
            left: '',
            right: '',
            marginLeft: '',
        });
    }
    else if (horizontalOrVertical === 'vertical') {
        // reset styles first
        $tripBlock.css({
            top: '',
            bottom: '',
            marginTop: '',
        });
    }

    var $sel = $(o.sel);
    var selWidth = $sel && $sel.outerWidth();
    var selHeight = $sel && $sel.outerHeight();
    var blockWidth = $tripBlock.outerWidth();
    var blockHeight = $tripBlock.outerHeight();
    var arrowHeight = 10;
    var arrowWidth = 10;
    var cssHorizontal = this.getIframeScrollLeft(o);
    var cssVertical = this.getIframeScrollTop(o);

    switch (o.position) {
      case 'screen-center':
        cssHorizontal = '50%';
        cssVertical = '50%';
        break;
      case 'screen-ne':
      case 'screen-se':
      case 'screen-nw':
      case 'screen-sw':
        cssHorizontal = TripConstant.TRIP_BLOCK_OFFSET_HORIZONTAL;
        cssVertical = TripConstant.TRIP_BLOCK_OFFSET_VERTICAL;
        break;
      case 'e':
        cssHorizontal += $sel.offset().left + selWidth + arrowWidth;
        cssVertical += $sel.offset().top - ((blockHeight - selHeight) / 2);
        break;
      case 's':
        cssHorizontal += $sel.offset().left + ((selWidth - blockWidth) / 2);
        cssVertical += $sel.offset().top + selHeight + arrowHeight;
        break;
      case 'w':
        cssHorizontal += $sel.offset().left - (arrowWidth + blockWidth);
        cssVertical += $sel.offset().top - ((blockHeight - selHeight) / 2);
        break;
      case 'n':
        /* falls through */
      default:
        cssHorizontal += $sel.offset().left + ((selWidth - blockWidth) / 2);
        cssVertical += $sel.offset().top - arrowHeight - blockHeight;
        break;
    }

    if (horizontalOrVertical === 'horizontal') {
      switch (o.position) {
        case 'screen-center':
          $tripBlock.css({
            left: cssHorizontal,
            marginLeft: -0.5 * blockWidth
          });
          break;
        case 'screen-se':
        case 'screen-ne':
          $tripBlock.css({
            right: cssHorizontal
          });
          break;
        case 'screen-sw':
        case 'screen-nw':
        case 'e':
        case 's':
        case 'w':
        case 'n':
          /* falls through */
        default:
          $tripBlock.css({
            left: cssHorizontal
          });
          break;
      }
    }
    else if (horizontalOrVertical === 'vertical') {

      switch (o.position) {
        case 'screen-center':
          $tripBlock.css({
            top: cssVertical,
            marginTop: -0.5 * blockHeight
          });
          break;
        case 'screen-sw':
        case 'screen-se':
          $tripBlock.css({
            bottom: cssVertical
          });
          break;
        case 'screen-nw':
        case 'screen-ne':
        case 'e':
        case 's':
        case 'w':
        case 'n':
          /* falls through */
        default:
          $tripBlock.css({
            top: cssVertical
          });
          break;
      }
    }
  },

  /**
   * Add animation on the trip block.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  addAnimation: function(o) {
    var animation = o.animation || this.settings.animation;
    if (TripAnimation.has(animation)) {
      this.$tripBlock.addClass('animated');
      this.$tripBlock.addClass(animation);
    }
  },

  /**
   * Remove animation from the trip block.
   *
   * @memberOf Trip
   * @type {Function}
   */
  removeAnimation: function() {
    this.$tripBlock.removeClass(TripAnimation.getAllInString());
    this.$tripBlock.removeClass('animated');
  },

  /**
   * After we positioned our trip block, we have to show it on the screen. If
   * the trip block is not on the screen, we will scroll the $root element and
   * then make sure it is definitely on the screen.
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  showTripBlock: function(o) {
    this.$tripBlock.css({
      display: 'inline-block',
      // we have to make it higher than the overlay
      zIndex: this.settings.overlayZindex + 1
    });

    var windowHeight = $(window).height();
    var windowTop = $(window).scrollTop();
    var frameTop = this.getIframeScrollTop(o);
    if (frameTop != 0) {
      frameTop -= windowTop;
    }
    var tripBlockTop = this.$tripBlock.offset().top + frameTop;
    var tripBlockHeight = this.$tripBlock.height();
    var OFFSET = 100; // make it look nice

    if (tripBlockTop + tripBlockHeight < windowTop + windowHeight &&
      tripBlockTop >= windowTop) {
        // tripBlock is located inside the current screen,
        // so we don't have to scroll
    }
    else {
      this.$root.animate({ scrollTop: tripBlockTop - OFFSET }, 'slow');
    }
  },

  /**
   * Return the scroll top offset needed in case the element is contained
   * in an iframe, else return 0
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  getIframeScrollTop: function(o) {
    if ($(o.sel).parents('html')[0] != this.$tripBlock.parents('html')[0]) {
      var offsetTop = 0;
      $(document).find('iframe').each(function(index, frame) {
        if ($(frame).contents().has($(o.sel))) {
          offsetTop = $(frame).offset().top;
          return false;
        }
      });
      return offsetTop - $(o.sel).parents('html,body').scrollTop();
    }
    return 0;
  },

  /**
   * Return the scroll left offset needed in case the element is contained
   * in an iframe, else return 0
   *
   * @memberOf Trip
   * @type {Function}
   * @param {Object} o
   */
  getIframeScrollLeft: function(o) {
    if ($(o.sel).parents('html')[0] != this.$tripBlock.parents('html')[0]) {
      var offsetLeft = 0;
      $(document).find('iframe').each(function(index, frame) {
        if ($(frame).contents().has($(o.sel))) {
          offsetLeft = $(frame).offset().left;
          return false;
        }
      });
      return offsetLeft - $(o.sel).parents('html,body').scrollLeft();
    }
    return 0;
  },

  /**
   * Hide the trip block.
   *
   * @memberOf Trip
   * @type {Function}
   */
  hideTripBlock: function() {
    this.removeAnimation();
    this.$tripBlock.fadeOut('slow');
  },

  /**
   * This is a method wrapper.
   *
   * @memberOf Trip
   * @type {Function}
   */
  create: function() {
    this.createTripBlock();
    this.createOverlay();
  },

  /**
   * This method is used to create a trip block at the first time when
   * start. If the trip block already exists on the DOM tree, we will
   * not create it again.
   *
   * @memberOf Trip
   * @type {Function}
   */
  createTripBlock: function() {
    // make sure the element doesn't exist in the DOM tree
    if (typeof $('.trip-block').get(0) === 'undefined') {
      var that = this;
      var tripBlockHTML = this.settings.tripBlockHTML;
      var $tripBlock = $(tripBlockHTML);

      $tripBlock
        .addClass(this.settings.tripTheme)
        .addClass(this.settings.tripClass)
        .addClass('tripjs');

      $('html>body').append($tripBlock);

      var $progressSteps = $tripBlock.find('.trip-progress-steps');
      if ($progressSteps) {
        var stepCache = [];
        var $step = $('<div class="trip-progress-step"></div>');

        for (var i = 0; i < this.tripData.length; i++) {
          stepCache.push($step.clone());
        }

        $progressSteps.append(stepCache);
      }

      $tripBlock.off('click.Trip');
      $tripBlock.on('click.Trip', function(e) {
        var tripObject = that.getCurrentTripObject();
        var toStopClickPropagation =
          TripUtils.isSet(tripObject.stopClickPropagation,
            that.settings.stopClickPropagation);

        if (toStopClickPropagation) {
          e.stopPropagation();
        }
      });

      var $closeButton = $tripBlock.find('.trip-close');
      if ($closeButton) {
        $closeButton.off('click.Trip');
        $closeButton.on('click.Trip', function(e) {
          e.preventDefault();
          var tripObject = that.getCurrentTripObject();
          var tripClose = tripObject.onTripClose || that.settings.onTripClose;
          tripClose.call(that, that.tripIndex, tripObject);
          that.stop();
        });
      }

      var $skipButton = $tripBlock.find('.trip-skip');
      if ($skipButton) {
        $skipButton.off('click.Trip');
        $skipButton.on('click.Trip', function(e) {
          e.preventDefault();
          var tripObject = that.getCurrentTripObject();
          var tripClose = tripObject.onTripClose || that.settings.onTripClose;
          tripClose.call(that, that.tripIndex, tripObject);
          that.stop();
        });
      }

      var $prevButton = $tripBlock.find('.trip-prev');
      if ($prevButton) {
        $prevButton.off('click.Trip');
        $prevButton.on('click', function(e) {
          e.preventDefault();
          // Force IE/FF to lose focus
          $(this).blur();
          that.prev();
        });
      }

      var $nextButton = $tripBlock.find('.trip-next');
      if ($nextButton) {
        $nextButton.off('click.Trip');
        $nextButton.on('click', function(e) {
          e.preventDefault();
          // Force IE/FF to lose focus
          $(this).blur();
          that.next();
        });
      }
    }
  },

  /**
   * This method is used to create overlay. If the overlay is in the DOM tree,
   * we will not create it again.
   *
   * @memberOf Trip
   * @type {Function}
   */
  createOverlay: function() {
    // make sure the element doesn't exist in the DOM tree
    if (typeof $('.trip-overlay').get(0) === 'undefined') {
      var html = [
        '<div class="trip-overlay">',
        '</div>'
      ].join('');

      var $overlay = $(html);
      $overlay
        .height($(window).height())
        .css({
          zIndex: this.settings.overlayZindex
        });

      $(this.settings.overlayHolder).append($overlay);
    }
  },

  /**
   * This is the handy function to toggle overlay.
   *
   * @memberOf Trip
   * @type {Function}
   */
  toggleExposedOverlay: function(toShow) {
    if (toShow) {
      this.$overlay.fadeIn();
    }
    else {
      this.$overlay.fadeOut();
    }
  },

  /**
   * Clean up all stuffs when we are going to start / restart a trip, so we
   * can make we won't mess up with old stuffs.
   *
   * @memberOf Trip
   * @type {Function}
   */
  cleanup: function() {
    $('.trip-overlay, .trip-block').remove();
  },

  /**
   * Initialize Trip.
   *
   * @memberOf Trip
   * @type {Function}
   */
  init: function() {
    if (this.settings.enableKeyBinding) {
      this.bindKeyEvents();
    }

    this.bindResizeEvents();

    // set refs
    this.$tripBlock = $('.trip-block');
    this.$bar = $('.trip-progress-bar');
    this.$overlay = $('.trip-overlay');

    this.setIndex(this.settings.tripIndex);
  },

  /**
   * Start Trip.
   *
   * @memberOf Trip
   * @type {Function}
   */
  start: function() {
    // cleanup old DOM first
    this.cleanup();

    // we will call this before initializing all stuffs
    this.settings.onStart.call(this);

    // create some necessary DOM elements at the first time like jQuery UI
    this.create();

    // init some necessary stuffs like events, late DOM refs
    // after creating DOMs
    this.init();

    // main entry
    this.run();
  }
};
